package de.xam.textsearch.fragment;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.xydra.index.IEntrySet;
import org.xydra.index.IMapSetIndex;
import org.xydra.index.ISerializableMapSetIndex;
import org.xydra.index.impl.FastEntrySetFactory;
import org.xydra.index.impl.SmallFastListBasedSet;
import org.xydra.index.impl.trie.SmallTrieStringMapSetIndex;
import org.xydra.index.iterator.ClosableIterator;
import org.xydra.index.iterator.IFilter;
import org.xydra.index.iterator.ITransformer;
import org.xydra.index.iterator.Iterators;
import org.xydra.index.query.Constraint;
import org.xydra.index.query.KeyEntryTuple;
import org.xydra.index.query.Pair;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

/**
 * A fragment is the smallest part of a text, phrase, word or token used for indexing!
 *
 * This class supports extensive log.trace.
 *
 * @author xamde
 * @param <V> value type
 */
public class FragmentIndex<V extends Serializable> implements ISerializableMapSetIndex<String, V> {

	private static final long serialVersionUID = 1L;

	private SmallTrieStringMapSetIndex<V> trie;

	public FragmentIndex() {
		// IMPROVE verify concurrent=true is required here
		this.trie = new SmallTrieStringMapSetIndex<V>(new FastEntrySetFactory<V>(true));
	}

	@Override
	public void clear() {
		this.trie.clear();
	}

	/**
	 * @param otherFuture the other map index is the future. What is found here and not present in this, has been
	 *        added. @NeverNull
	 * @return an IMapSetDiff
	 */
	@Override
	public IMapSetDiff<String, V> computeDiff(final IMapSetIndex<String, V> otherFuture) {
		return this.trie.computeDiff(otherFuture);
	}

	/**
	 * @param keyConstraint @NeverNull be careful in what you put here, this is closely tied to internal algorithms
	 * @return an ClosableIterator that ranges over all entries indexes by keys, where the keys match keyConstraint.
	 *         Close it to release internal read-locks.
	 */
	@Override
	public ClosableIterator<V> constraintIterator(final Constraint<String> keyConstraint) {
		assert keyConstraint != null;
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.constraintIterator with keyConstraint=" + keyConstraint);
		}
		return this.trie.constraintIterator(keyConstraint);
	}

	/**
	 * @param keyConstraint @NeverNull be careful in what you put here, this is closely tied to internal algorithms
	 * @param entryConstraint
	 * @return true iff this index contains a tuple matching the given constraints.
	 */
	@Override
	public boolean contains(final Constraint<String> keyConstraint, final Constraint<V> entryConstraint) {
		final boolean result = this.trie.contains(keyConstraint, entryConstraint);
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.contains keyConstraint=" + keyConstraint + "; entryConstraint=" + entryConstraint
					+ " ? => " + result);
		}
		return result;
	}

	@Override
	public boolean contains(final String key, final V entry) {
		final boolean result = this.trie.contains(key, entry);
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.contains key=" + key + "; entry=" + entry + " ? => " + result);
		}
		return result;
	}

	/**
	 * @param key
	 * @NeverNull
	 * @return true if this index contains any tuple (key,*)
	 */
	@Override
	public boolean containsKey(final String key) {
		final boolean result = this.trie.containsKey(key);
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.containsKey='" + key + "' ? => " + result);
		}
		return result;
	}

	/**
	 * De-index all current entries with (key1, *).
	 *
	 * @param key1
	 * @NeverNull
	 */
	@Override
	public boolean deIndex(final String key1) {
		return this.trie.deIndex(key1);
	}

	/**
	 * Removed a tuple from the index.
	 *
	 * @param key1
	 * @NeverNull
	 * @param entry
	 * @NeverNull
	 * @return true iff set K contained entry
	 */
	@Override
	public boolean deIndex(final String key1, final V entry) {
		return this.trie.deIndex(key1, entry);
	}

	/**
	 * Add a tuple to the index
	 *
	 * @param key1
	 * @NeverNull
	 * @param entry
	 * @NeverNull
	 * @return true iff set K did not contain entry yet
	 */
	@Override
	public boolean index(final String key1, final V entry) {
		return this.trie.index(key1, entry);
	}

	@Override
	public boolean isEmpty() {
		return this.trie.isEmpty();
	}

	/**
	 * @return an iterator over all keys ?x in tuples (?x,?y) without duplicates.
	 */
	@Override
	public ClosableIterator<String> keyIterator() {
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.keyIterator");
		}
		return this.trie.keyIterator();
	}

	/**
	 * @param key
	 * @NeverNull
	 * @return an {@link IEntrySet} containing all ?y from tuples (key,?y), @CanBeNull if no entries
	 */
	@Override
	public IEntrySet<V> lookup(final String key) {
		final IEntrySet<V> result = this.trie.lookup(key);
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.lookup '" + key + "' = " + result);
		}
		return result;
	}

	/**
	 * @param keyConstraint constraint on the key @NeverNull
	 * @param entryConstraint constraint on the value @NeverNull
	 * @return an iterator over all result tuples matching the constraints @NeverNull
	 */
	@Override
	public ClosableIterator<KeyEntryTuple<String, V>> tupleIterator(final Constraint<String> keyConstraint,
			final Constraint<V> entryConstraint) {
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.tupleIterator");
		}
		return this.trie.tupleIterator(keyConstraint, entryConstraint);
	}

	private static final Logger log = LoggerFactory.getLogger(FragmentIndex.class);

	public ClosableIterator<V> executeQuery(final FragmentQuery<V> fragmentQuery) {
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.query " + fragmentQuery);
		}

		final ClosableIterator<V> it = this.trie.valueIterator(fragmentQuery, fragmentQuery.getOptionalValueFilter());
		if (fragmentQuery.getMaxResults() < Integer.MAX_VALUE) {
			return Iterators.limit(it, fragmentQuery.getMaxResults());
		} else {
			return it;
		}
	}

	/**
	 * Materialize entrySet while respecting filters of query to a result set.
	 *
	 * See also {@link #filterAndLimit(Set, FragmentQuery)}
	 *
	 * @param entrySet @NeverNull
	 * @param fragmentQuery for configuration details of the query @NeverNull
	 * @param result where to add the result @NeverNull
	 */
	private static <V extends Serializable> void addAllWithFilterAndLimit(final IEntrySet<V> entrySet,
			final FragmentQuery<V> fragmentQuery, final Set<V> result) {
		Iterator<V> it = entrySet.iterator();
		if (fragmentQuery.isFiltered()) {
			it = Iterators.filter(it, fragmentQuery.getOptionalValueFilter());
		}
		if (fragmentQuery.isLimited()) {
			int spaceLeft = fragmentQuery.getMaxResults() - result.size();
			while (it.hasNext() && spaceLeft > 0) {
				result.add(it.next());
				spaceLeft--;
			}
		} else {
			Iterators.addAll(it, result);
		}
	}

	/**
	 * See also {@link #addAllWithFilterAndLimit(IEntrySet, FragmentQuery, Set)}
	 *
	 * @param set
	 * @param fragmentQuery
	 * @return
	 */
	private static <V extends Serializable> Set<V> toSetWithFilterAndLimit(final IEntrySet<V> entrySet,
			final FragmentQuery<V> fragmentQuery) {
		if (!fragmentQuery.isFiltered() && !fragmentQuery.isLimited()) {
			return entrySet.toSet();
		} else {
			final Set<V> result = new SmallFastListBasedSet<V>(entrySet.size());
			addAllWithFilterAndLimit(entrySet, fragmentQuery, result);
			return result;
		}
	}

	/**
	 * @param fragmentQuery
	 * @return a single Java set
	 */
	public Set<V> executeQueryAsSet(final FragmentQuery<V> fragmentQuery) {
		final ClosableIterator<IEntrySet<V>> entrySetIt = this.trie.valueAsEntrySetIterator(fragmentQuery);
		Set<V> result;
		IEntrySet<V> entrySet = entrySetIt.next();
		if (entrySet == null) {
			// fast path
			entrySetIt.close();
			result = Collections.emptySet();
		} else if (!entrySetIt.hasNext()) {
			// use single result set
			entrySetIt.close();
			result = toSetWithFilterAndLimit(entrySet, fragmentQuery);
		} else {
			// else: slow path: merge sets
			result = new HashSet<V>();
			do {
				addAllWithFilterAndLimit(entrySet, fragmentQuery, result);
				entrySet = entrySetIt.next();
			} while (entrySetIt.hasNext());
			entrySetIt.close();
		}

		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.query " + fragmentQuery + " => " + result);
		}

		return result;
	}

	/**
	 * maxResults understood as maximal number of <em>sets</em> to return
	 *
	 * @param fragmentQuery
	 * @return matching IEntrySet objects, can return nulls in the iterator
	 */
	public ClosableIterator<Set<V>> executeQueryAsSets(final FragmentQuery<V> fragmentQuery) {
		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.executeQueryAsSets " + fragmentQuery);
		}

		final ClosableIterator<IEntrySet<V>> entrySetIt = this.trie.valueAsEntrySetIterator(fragmentQuery);
		ClosableIterator<Set<V>> setIt = Iterators.transform(entrySetIt, new ITransformer<IEntrySet<V>, Set<V>>() {

			@Override
			public Set<V> transform(final IEntrySet<V> entrySet) {

				if (fragmentQuery.getOptionalValueFilter() == null && !fragmentQuery.isLimited()) {
					return entrySet.toSet();
				} else {
					// filter
					Iterator<V> it = entrySet.iterator();
					if (fragmentQuery.getOptionalValueFilter() != null) {
						it = Iterators.filter(it, fragmentQuery.getOptionalValueFilter());
					}
					if (!it.hasNext()) {
						return null;
					}

					final Set<V> set = new HashSet<V>();
					Iterators.addAll(it, set);
					return set;
				}
			}

		});
		/* filter out nulls which we might just have introduced with the filter */
		setIt = Iterators.filter(setIt, new IFilter<Set<V>>() {

			@Override
			public boolean matches(final Set<V> entry) {
				return entry != null;
			}
		});

		if (fragmentQuery.isLimited()) {
			setIt = Iterators.limit(setIt, fragmentQuery.getMaxResults());
		}

		return setIt;
	}

	FragmentQuery<V> createQuery(final String fragment, final boolean prefixMatch) {
		return FragmentQuery.<V> create(this, fragment, prefixMatch);
	}

	public void dump() {
		this.trie.dump();
	}

	/**
	 * @param s
	 * @param start
	 * @return a pair: 1) the number of characters, starting from start, are the longest match so that at least one
	 *         result is returned; 2) the entry set of result values
	 */
	public Pair<Integer, Set<V>> getLongestMatch(final String s, final int start) {
		final Pair<Integer, Set<V>> result = this.trie.getLongestMatch(s, start);

		if (log.isTraceEnabled()) {
			log.trace("FragmentIndex.longestMatch for '" + s + "' start:" + start + " =" + result);
		}

		return result;
	}

	public String toDebugString() {
		return this.trie.toDebugString();
	}

	@Override
	public String toString(final String indent) {
		return indent + this.trie.toDebugString();
	}

	/**
	 * @param indexState
	 */
	public void setInternalIndexState(final SmallTrieStringMapSetIndex<V> indexState) {
		this.trie = indexState;
	}

	public SmallTrieStringMapSetIndex<V> getInternalIndexState() {
		return this.trie;
	}

	public int size() {
		return this.trie.size();
	}
}
