package de.xam.textsearch;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

/**
 * A match has a Value object and a set of matched tokens.
 *
 * Natural sort order is by quality score.
 *
 * @author xamde
 *
 * @param <V>
 */
public class Match<V> implements Serializable {

	private static final long serialVersionUID = 1L;

	@Override
	public String toString() {
		return "Match for '" + this.value + "' "

		+ "q=" + this.q() + " strings=" + this.matchedStrings;
	}

	/** Original casing, distinct matches, no duplicates */
	private final Set<String> matchedStrings = new HashSet<String>();

	/**
	 * @return
	 */
	public Set<String> getMatchedStrings() {
		return this.matchedStrings;
	}

	/**
	 * @param value @NeverNull
	 */
	public Match(final V value) {
		super();
		assert value != null;
		this.value = value;
	}

	/**
	 * @param value @NeverNull
	 * @param q
	 */
	public Match(final V value, final double q) {
		this(value);
		this.q = q;
	}

	private V value;

	/**
	 * More is better
	 */
	public double q = -1;

	/**
	 * @return the quality value; >= 0; unbounded.
	 */
	public double q() {
		assert this.q >= 0 : "q="+this.q;
		return this.q;

	}

	/**
	 * @return the value of the match, i.e. the id of the underlying object
	 */
	public V getValue() {
		return this.value;
	}

	/**
	 * @param q
	 */
	public void setQuality(final double q) {
		this.q = q;
	}

	/**
	 * @param s
	 */
	public void addMatchedString(final String s) {
		this.matchedStrings.add(s);
	}

	/**
	 * @param s
	 * @return
	 */
	public boolean hasMatchedString(final String s) {
		return this.matchedStrings.contains(s);
	}

	@Override
	public int hashCode() {
		return this.value.hashCode();
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		@SuppressWarnings("unchecked") final Match<V> other = (Match<V>) obj;
		return other.value.equals(this.value);
	}

}
