package de.xam.texthtml.text;

import org.xydra.annotations.RunsInGWT;
import org.xydra.annotations.Template;
import org.xydra.base.id.XidCodec;
import org.xydra.core.serialize.json.JsonEncoder;

import de.xam.texthtml.html.SharedHtmlUtils;

/**
 * Used on client and server side to produce safe HTML.
 *
 * @author xamde
 */
@RunsInGWT(true)
public class EncTool {

	private static final String XSS_INFO_URL = "http://en.wikipedia.org/wiki/Cross-site_scripting";

	@Template("Exposed as $enc.htmlText(..)")
	public static String htmlText(final String s) {
		return SharedHtmlUtils.sanitize(s);
	}

	// TODO Tristan
	public static String htmlHref(final String unsafeHref) {
		final String lower = unsafeHref.toLowerCase();
		if (lower.startsWith("http:") || lower.startsWith("https:") || lower.startsWith("mailto:")) {
			return unsafeHref;
		} else {
			return XSS_INFO_URL;
		}
	}

	private static EncTool instance;

	public static synchronized EncTool getInstance() {
		if (instance == null) {
			instance = new EncTool();
		}
		return instance;
	}

	public static String jsonString(final String unsafe) {
		return JsonEncoder.encode(unsafe);
	}

	/**
	 * @param unsafe
	 * @return any string as a valid JSON string embedded in a HTML host page
	 */
	public static String jsonInHtml(final String unsafe) {
		final String html = htmlText(unsafe);
		return JsonEncoder.encode(html);
	}

	public static String htmlTextAutolinkUrls(final String unsafePlaintext) {
		return TextRenderer.renderAsSafeHtml(unsafePlaintext);
	}

	/**
	 * Escape to make it a valid CSS id or class name
	 *
	 * @param raw
	 * @return all dangerous characters escaped as '\X ' with x = 2-6 hex characters.
	 */
	public static String escapeCssIdOrClassname(final String raw) {
		assert raw != null;

		final StringBuilder esc = new StringBuilder();
		int i = 0;
		while (i < raw.length()) {
			final int c = raw.codePointAt(i);
			i += Character.charCount(c);

			if (c == '-'

					|| c == '_'

					|| c >= 'a' && c <= 'z'

					|| c >= 'A' && c <= 'Z'

					|| c >= '0' && c <= '9'

					) {
				esc.appendCodePoint(c);
			} else {
				esc.append('\\');
				esc.append(Integer.toHexString(c));
				esc.append(" ");
			}

		}

		return esc.toString();
	}

	/**
	 * according to https://en.wikipedia.org/wiki/Filename#Reserved_characters_and_words
	 */
	public static final char[] FILENAME_CRITICAL = { '/', '\\', '?', '%', '*', ':', '|', '\"', '<', '>', ' ' };

	/**
	 * @param name a local name
	 * @return a valid name on Windows, Linux and OS X
	 */
	public static String escapeFilename(final String name) {
		final StringBuilder enc = new StringBuilder();
		int i = 0;
		while (i < name.length()) {
			final int codePoint = name.codePointAt(i);
			if (codePoint == '%' || TextTool.isOneOf(codePoint, FILENAME_CRITICAL)) {
				XidCodec.appendEncoded(enc, '%', codePoint, 200);
			} else {
				enc.appendCodePoint(codePoint);
			}

			i += Character.charCount(codePoint);
		}

		return enc.toString();
	}

	public static String unescapeFilename(final String filename) {
		return XidCodec.decode(filename, '%');
	}

	final private static char[] hexArray = "0123456789ABCDEF".toCharArray();

	/**
	 * Convert a byte array to hex string, pretty fast implementation.
	 *
	 * @param bytes
	 * @return hex string
	 */
	public static String bytesToHex(final byte[] bytes) {
		if (bytes == null) {
			return null;
		}

		final char[] hexChars = new char[bytes.length * 2];
		for (int j = 0; j < bytes.length; j++) {
			final int v = bytes[j] & 0xFF;
			hexChars[j * 2] = hexArray[v >>> 4];
			hexChars[j * 2 + 1] = hexArray[v & 0x0F];
		}
		return new String(hexChars);
	}

}
