package de.xam.p13n.shared.time;

import java.util.HashSet;
import java.util.Set;

import org.xydra.annotations.RequiresAppEngine;
import org.xydra.annotations.RunsInAppEngine;
import org.xydra.annotations.RunsInGWT;

/**
 * A portable version of the zone.tab file in the tz database. This file
 * contains only entries with time zone names, but not the actual UTC offsets.
 *
 * Note: This class should not be bundled in JS unless really needed :-)
 *
 * @author xamde
 *
 */
@RunsInAppEngine(true)
@RunsInGWT(true)
@RequiresAppEngine(false)
public class TzDatabase {

	public static class TzEntry {

		/**
		 * Comments; present if and only if the country has multiple rows. Null
		 * otherwise.
		 */
		public final String comment;

		/**
		 * ISO 3166 2-character country code. See the file `iso3166.tab'.
		 */
		public final String countryCode;

		/**
		 * Latitude and longitude of the zone's principal location in ISO 6709
		 * sign-degrees-minutes-seconds format, either +-DDMM+-DDDMM or
		 * +-DDMMSS+-DDDMMSS, first latitude (+ is north), then longitude (+ is
		 * east).
		 */
		public final String latLong;
		/**
		 * Zone name used in value of TZ environment variable.
		 */
		public final String zoneName;

		public TzEntry(final String countryCode, final String latLong, final String zoneName) {
			this.countryCode = countryCode;
			this.latLong = latLong;
			this.zoneName = zoneName;
			this.comment = null;
		}

		/**
		 * @param countryCode
		 *            ISO
		 * @param latLong
		 *            TODO 2012-03 document syntax for latlong in TzDatabase
		 * @param zoneName
		 *            Java/Olson standard
		 * @param comment
		 *            optional
		 */
		public TzEntry(final String countryCode, final String latLong, final String zoneName, final String comment) {
			this.countryCode = countryCode;
			this.latLong = latLong;
			this.zoneName = zoneName;
			this.comment = comment;
		}

	}

	public static final TzEntry[] DATABASE = {
			new TzEntry("AD", "+4230+00131", "Europe/Andorra"),
			new TzEntry("AE", "+2518+05518", "Asia/Dubai"),
			new TzEntry("AF", "+3431+06912", "Asia/Kabul"),
			new TzEntry("AG", "+1703-06148", "America/Antigua"),
			new TzEntry("AI", "+1812-06304", "America/Anguilla"),
			new TzEntry("AL", "+4120+01950", "Europe/Tirane"),
			new TzEntry("AM", "+4011+04430", "Asia/Yerevan"),
			new TzEntry("AN", "+1211-06900", "America/Curacao"),
			new TzEntry("AO", "-0848+01314", "Africa/Luanda"),
			new TzEntry("AQ", "-7750+16636", "Antarctica/McMurdo", "McMurdo Station, Ross Island"),
			new TzEntry("AQ", "-9000+00000", "Antarctica/South_Pole",
					"Amundsen-Scott Station, South Pole"),
			new TzEntry("AQ", "-6734-06808", "Antarctica/Rothera",
					"Rothera Station, Adelaide Island"),
			new TzEntry("AQ", "-6448-06406", "Antarctica/Palmer", "Palmer Station, Anvers Island"),
			new TzEntry("AQ", "-6736+06253", "Antarctica/Mawson", "Mawson Station, Holme Bay"),
			new TzEntry("AQ", "-6835+07758", "Antarctica/Davis", "Davis Station, Vestfold Hills"),
			new TzEntry("AQ", "-6617+11031", "Antarctica/Casey", "Casey Station, Bailey Peninsula"),
			new TzEntry("AQ", "-7824+10654", "Antarctica/Vostok", "Vostok Station, Lake Vostok"),
			new TzEntry("AQ", "-6640+14001", "Antarctica/DumontDUrville",
					"Dumont-d'Urville Station, Terre Adelie"),
			new TzEntry("AQ", "-690022+0393524", "Antarctica/Syowa", "Syowa Station, E Ongul I"),
			new TzEntry("AQ", "-5430+15857", "Antarctica/Macquarie",
					"Macquarie Island Station, Macquarie Island"),
			new TzEntry("AR", "-3436-05827", "America/Argentina/Buenos_Aires",
					"Buenos Aires (BA, CF)"),
			new TzEntry("AR", "-3124-06411", "America/Argentina/Cordoba",
					"most locations (CB, CC, CN, ER, FM, MN, SE, SF)"),
			new TzEntry("AR", "-2447-06525", "America/Argentina/Salta", "(SA, LP, NQ, RN)"),
			new TzEntry("AR", "-2411-06518", "America/Argentina/Jujuy", "Jujuy (JY)"),
			new TzEntry("AR", "-2649-06513", "America/Argentina/Tucuman", "Tucuman (TM)"),
			new TzEntry("AR", "-2828-06547", "America/Argentina/Catamarca",
					"Catamarca (CT), Chubut (CH)"),
			new TzEntry("AR", "-2926-06651", "America/Argentina/La_Rioja", "La Rioja (LR)"),
			new TzEntry("AR", "-3132-06831", "America/Argentina/San_Juan", "San Juan (SJ)"),
			new TzEntry("AR", "-3253-06849", "America/Argentina/Mendoza", "Mendoza (MZ)"),
			new TzEntry("AR", "-3319-06621", "America/Argentina/San_Luis", "San Luis (SL)"),
			new TzEntry("AR", "-5138-06913", "America/Argentina/Rio_Gallegos", "Santa Cruz (SC)"),
			new TzEntry("AR", "-5448-06818", "America/Argentina/Ushuaia", "Tierra del Fuego (TF)"),
			new TzEntry("AS", "-1416-17042", "Pacific/Pago_Pago"),
			new TzEntry("AT", "+4813+01620", "Europe/Vienna"),
			new TzEntry("AU", "-3133+15905", "Australia/Lord_Howe", "Lord Howe Island"),
			new TzEntry("AU", "-4253+14719", "Australia/Hobart", "Tasmania - most locations"),
			new TzEntry("AU", "-3956+14352", "Australia/Currie", "Tasmania - King Island"),
			new TzEntry("AU", "-3749+14458", "Australia/Melbourne", "Victoria"),
			new TzEntry("AU", "-3352+15113", "Australia/Sydney", "New South Wales - most locations"),
			new TzEntry("AU", "-3157+14127", "Australia/Broken_Hill",
					"New South Wales - Yancowinna"),
			new TzEntry("AU", "-2728+15302", "Australia/Brisbane", "Queensland - most locations"),
			new TzEntry("AU", "-2016+14900", "Australia/Lindeman", "Queensland - Holiday Islands"),
			new TzEntry("AU", "-3455+13835", "Australia/Adelaide", "South Australia"),
			new TzEntry("AU", "-1228+13050", "Australia/Darwin", "Northern Territory"),
			new TzEntry("AU", "-3157+11551", "Australia/Perth",
					"Western Australia - most locations"),
			new TzEntry("AU", "-3143+12852", "Australia/Eucla", "Western Australia - Eucla area"),
			new TzEntry("AW", "+1230-06958", "America/Aruba"),
			new TzEntry("AX", "+6006+01957", "Europe/Mariehamn"),
			new TzEntry("AZ", "+4023+04951", "Asia/Baku"),
			new TzEntry("BA", "+4352+01825", "Europe/Sarajevo"),
			new TzEntry("BB", "+1306-05937", "America/Barbados"),
			new TzEntry("BD", "+2343+09025", "Asia/Dhaka"),
			new TzEntry("BE", "+5050+00420", "Europe/Brussels"),
			new TzEntry("BF", "+1222-00131", "Africa/Ouagadougou"),
			new TzEntry("BG", "+4241+02319", "Europe/Sofia"),
			new TzEntry("BH", "+2623+05035", "Asia/Bahrain"),
			new TzEntry("BI", "-0323+02922", "Africa/Bujumbura"),
			new TzEntry("BJ", "+0629+00237", "Africa/Porto-Novo"),
			new TzEntry("BL", "+1753-06251", "America/St_Barthelemy"),
			new TzEntry("BM", "+3217-06446", "Atlantic/Bermuda"),
			new TzEntry("BN", "+0456+11455", "Asia/Brunei"),
			new TzEntry("BO", "-1630-06809", "America/La_Paz"),
			new TzEntry("BR", "-0351-03225", "America/Noronha", "Atlantic islands"),
			new TzEntry("BR", "-0127-04829", "America/Belem", "Amapa, E Para"),
			new TzEntry("BR", "-0343-03830", "America/Fortaleza", "NE Brazil (MA, PI, CE, RN, PB)"),
			new TzEntry("BR", "-0803-03454", "America/Recife", "Pernambuco"),
			new TzEntry("BR", "-0712-04812", "America/Araguaina", "Tocantins"),
			new TzEntry("BR", "-0940-03543", "America/Maceio", "Alagoas, Sergipe"),
			new TzEntry("BR", "-1259-03831", "America/Bahia", "Bahia"),
			new TzEntry("BR", "-2332-04637", "America/Sao_Paulo",
					"S & SE Brazil (GO, DF, MG, ES, RJ, SP, PR, SC, RS)"),
			new TzEntry("BR", "-2027-05437", "America/Campo_Grande", "Mato Grosso do Sul"),
			new TzEntry("BR", "-1535-05605", "America/Cuiaba", "Mato Grosso"),
			new TzEntry("BR", "-0226-05452", "America/Santarem", "W Para"),
			new TzEntry("BR", "-0846-06354", "America/Porto_Velho", "Rondonia"),
			new TzEntry("BR", "+0249-06040", "America/Boa_Vista", "Roraima"),
			new TzEntry("BR", "-0308-06001", "America/Manaus", "E Amazonas"),
			new TzEntry("BR", "-0640-06952", "America/Eirunepe", "W Amazonas"),
			new TzEntry("BR", "-0958-06748", "America/Rio_Branco", "Acre"),
			new TzEntry("BS", "+2505-07721", "America/Nassau"),
			new TzEntry("BT", "+2728+08939", "Asia/Thimphu"),
			new TzEntry("BW", "-2439+02555", "Africa/Gaborone"),
			new TzEntry("BY", "+5354+02734", "Europe/Minsk"),
			new TzEntry("BZ", "+1730-08812", "America/Belize"),
			new TzEntry("CA", "+4734-05243", "America/St_Johns",
					"Newfoundland Time, including SE Labrador"),
			new TzEntry("CA", "+4439-06336", "America/Halifax",
					"Atlantic Time - Nova Scotia (most places), PEI"),
			new TzEntry("CA", "+4612-05957", "America/Glace_Bay",
					"Atlantic Time - Nova Scotia - places that did not observe DST 1966-1971"),
			new TzEntry("CA", "+4606-06447", "America/Moncton", "Atlantic Time - New Brunswick"),
			new TzEntry("CA", "+5320-06025", "America/Goose_Bay",
					"Atlantic Time - Labrador - most locations"),
			new TzEntry("CA", "+5125-05707", "America/Blanc-Sablon",
					"Atlantic Standard Time - Quebec - Lower North Shore"),
			new TzEntry("CA", "+4531-07334", "America/Montreal",
					"Eastern Time - Quebec - most locations"),
			new TzEntry("CA", "+4339-07923", "America/Toronto",
					"Eastern Time - Ontario - most locations"),
			new TzEntry("CA", "+4901-08816", "America/Nipigon",
					"Eastern Time - Ontario & Quebec - places that did not observe DST 1967-1973"),
			new TzEntry("CA", "+4823-08915", "America/Thunder_Bay",
					"Eastern Time - Thunder Bay, Ontario"),
			new TzEntry("CA", "+6344-06828", "America/Iqaluit",
					"Eastern Time - east Nunavut - most locations"),
			new TzEntry("CA", "+6608-06544", "America/Pangnirtung",
					"Eastern Time - Pangnirtung, Nunavut"),
			new TzEntry("CA", "+744144-0944945", "America/Resolute",
					"Eastern Standard Time - Resolute, Nunavut"),
			new TzEntry("CA", "+484531-0913718", "America/Atikokan",
					"Eastern Standard Time - Atikokan, Ontario and Southampton I, Nunavut"),
			new TzEntry("CA", "+624900-0920459", "America/Rankin_Inlet",
					"Central Time - central Nunavut"),
			new TzEntry("CA", "+4953-09709", "America/Winnipeg",
					"Central Time - Manitoba & west Ontario"),
			new TzEntry("CA", "+4843-09434", "America/Rainy_River",
					"Central Time - Rainy River & Fort Frances, Ontario"),
			new TzEntry("CA", "+5024-10439", "America/Regina",
					"Central Standard Time - Saskatchewan - most locations"),
			new TzEntry("CA", "+5017-10750", "America/Swift_Current",
					"Central Standard Time - Saskatchewan - midwest"),
			new TzEntry("CA", "+5333-11328", "America/Edmonton",
					"Mountain Time - Alberta, east British Columbia & west Saskatchewan"),
			new TzEntry("CA", "+690650-1050310", "America/Cambridge_Bay",
					"Mountain Time - west Nunavut"),
			new TzEntry("CA", "+6227-11421", "America/Yellowknife",
					"Mountain Time - central Northwest Territories"),
			new TzEntry("CA", "+682059-1334300", "America/Inuvik",
					"Mountain Time - west Northwest Territories"),
			new TzEntry("CA", "+5946-12014", "America/Dawson_Creek",
					"Mountain Standard Time - Dawson Creek & Fort Saint John, British Columbia"),
			new TzEntry("CA", "+4916-12307", "America/Vancouver",
					"Pacific Time - west British Columbia"),
			new TzEntry("CA", "+6043-13503", "America/Whitehorse", "Pacific Time - south Yukon"),
			new TzEntry("CA", "+6404-13925", "America/Dawson", "Pacific Time - north Yukon"),
			new TzEntry("CC", "-1210+09655", "Indian/Cocos"),
			new TzEntry("CD", "-0418+01518", "Africa/Kinshasa", "west Dem. Rep. of Congo"),
			new TzEntry("CD", "-1140+02728", "Africa/Lubumbashi", "east Dem. Rep. of Congo"),
			new TzEntry("CF", "+0422+01835", "Africa/Bangui"),
			new TzEntry("CG", "-0416+01517", "Africa/Brazzaville"),
			new TzEntry("CH", "+4723+00832", "Europe/Zurich"),
			new TzEntry("CI", "+0519-00402", "Africa/Abidjan"),
			new TzEntry("CK", "-2114-15946", "Pacific/Rarotonga"),
			new TzEntry("CL", "-3327-07040", "America/Santiago", "most locations"),
			new TzEntry("CL", "-2709-10926", "Pacific/Easter", "Easter Island & Sala y Gomez"),
			new TzEntry("CM", "+0403+00942", "Africa/Douala"),
			new TzEntry("CN", "+3114+12128", "Asia/Shanghai",
					"east China - Beijing, Guangdong, Shanghai, etc."),
			new TzEntry("CN", "+4545+12641", "Asia/Harbin", "Heilongjiang (except Mohe), Jilin"),
			new TzEntry("CN", "+2934+10635", "Asia/Chongqing",
					"central China - Sichuan, Yunnan, Guangxi, Shaanxi, Guizhou, etc."),
			new TzEntry("CN", "+4348+08735", "Asia/Urumqi", "most of Tibet & Xinjiang"),
			new TzEntry("CN", "+3929+07559", "Asia/Kashgar", "west Tibet & Xinjiang"),
			new TzEntry("CO", "+0436-07405", "America/Bogota"),
			new TzEntry("CR", "+0956-08405", "America/Costa_Rica"),
			new TzEntry("CU", "+2308-08222", "America/Havana"),
			new TzEntry("CV", "+1455-02331", "Atlantic/Cape_Verde"),
			new TzEntry("CX", "-1025+10543", "Indian/Christmas"),
			new TzEntry("CY", "+3510+03322", "Asia/Nicosia"),
			new TzEntry("CZ", "+5005+01426", "Europe/Prague"),
			new TzEntry("DE", "+5230+01322", "Europe/Berlin"),
			new TzEntry("DJ", "+1136+04309", "Africa/Djibouti"),
			new TzEntry("DK", "+5540+01235", "Europe/Copenhagen"),
			new TzEntry("DM", "+1518-06124", "America/Dominica"),
			new TzEntry("DO", "+1828-06954", "America/Santo_Domingo"),
			new TzEntry("DZ", "+3647+00303", "Africa/Algiers"),
			new TzEntry("EC", "-0210-07950", "America/Guayaquil", "mainland"),
			new TzEntry("EC", "-0054-08936", "Pacific/Galapagos", "Galapagos Islands"),
			new TzEntry("EE", "+5925+02445", "Europe/Tallinn"),
			new TzEntry("EG", "+3003+03115", "Africa/Cairo"),
			new TzEntry("EH", "+2709-01312", "Africa/El_Aaiun"),
			new TzEntry("ER", "+1520+03853", "Africa/Asmara"),
			new TzEntry("ES", "+4024-00341", "Europe/Madrid", "mainland"),
			new TzEntry("ES", "+3553-00519", "Africa/Ceuta", "Ceuta & Melilla"),
			new TzEntry("ES", "+2806-01524", "Atlantic/Canary", "Canary Islands"),
			new TzEntry("ET", "+0902+03842", "Africa/Addis_Ababa"),
			new TzEntry("FI", "+6010+02458", "Europe/Helsinki"),
			new TzEntry("FJ", "-1808+17825", "Pacific/Fiji"),
			new TzEntry("FK", "-5142-05751", "Atlantic/Stanley"),
			new TzEntry("FM", "+0725+15147", "Pacific/Chuuk", "Chuuk (Truk) and Yap"),
			new TzEntry("FM", "+0658+15813", "Pacific/Pohnpei", "Pohnpei (Ponape)"),
			new TzEntry("FM", "+0519+16259", "Pacific/Kosrae", "Kosrae"),
			new TzEntry("FO", "+6201-00646", "Atlantic/Faroe"),
			new TzEntry("FR", "+4852+00220", "Europe/Paris"),
			new TzEntry("GA", "+0023+00927", "Africa/Libreville"),
			new TzEntry("GB", "+513030-0000731", "Europe/London"),
			new TzEntry("GD", "+1203-06145", "America/Grenada"),
			new TzEntry("GE", "+4143+04449", "Asia/Tbilisi"),
			new TzEntry("GF", "+0456-05220", "America/Cayenne"),
			new TzEntry("GG", "+4927-00232", "Europe/Guernsey"),
			new TzEntry("GH", "+0533-00013", "Africa/Accra"),
			new TzEntry("GI", "+3608-00521", "Europe/Gibraltar"),
			new TzEntry("GL", "+6411-05144", "America/Godthab", "most locations"),
			new TzEntry("GL", "+7646-01840", "America/Danmarkshavn",
					"east coast, north of Scoresbysund"),
			new TzEntry("GL", "+7029-02158", "America/Scoresbysund",
					"Scoresbysund / Ittoqqortoormiit"),
			new TzEntry("GL", "+7634-06847", "America/Thule", "Thule / Pituffik"),
			new TzEntry("GM", "+1328-01639", "Africa/Banjul"),
			new TzEntry("GN", "+0931-01343", "Africa/Conakry"),
			new TzEntry("GP", "+1614-06132", "America/Guadeloupe"),
			new TzEntry("GQ", "+0345+00847", "Africa/Malabo"),
			new TzEntry("GR", "+3758+02343", "Europe/Athens"),
			new TzEntry("GS", "-5416-03632", "Atlantic/South_Georgia"),
			new TzEntry("GT", "+1438-09031", "America/Guatemala"),
			new TzEntry("GU", "+1328+14445", "Pacific/Guam"),
			new TzEntry("GW", "+1151-01535", "Africa/Bissau"),
			new TzEntry("GY", "+0648-05810", "America/Guyana"),
			new TzEntry("HK", "+2217+11409", "Asia/Hong_Kong"),
			new TzEntry("HN", "+1406-08713", "America/Tegucigalpa"),
			new TzEntry("HR", "+4548+01558", "Europe/Zagreb"),
			new TzEntry("HT", "+1832-07220", "America/Port-au-Prince"),
			new TzEntry("HU", "+4730+01905", "Europe/Budapest"),
			new TzEntry("ID", "-0610+10648", "Asia/Jakarta", "Java & Sumatra"),
			new TzEntry("ID", "-0002+10920", "Asia/Pontianak", "west & central Borneo"),
			new TzEntry("ID", "-0507+11924", "Asia/Makassar",
					"east & south Borneo, Celebes, Bali, Nusa Tengarra, west Timor"),
			new TzEntry("ID", "-0232+14042", "Asia/Jayapura", "Irian Jaya & the Moluccas"),
			new TzEntry("IE", "+5320-00615", "Europe/Dublin"),
			new TzEntry("IL", "+3146+03514", "Asia/Jerusalem"),
			new TzEntry("IM", "+5409-00428", "Europe/Isle_of_Man"),
			new TzEntry("IN", "+2232+08822", "Asia/Kolkata"),
			new TzEntry("IO", "-0720+07225", "Indian/Chagos"),
			new TzEntry("IQ", "+3321+04425", "Asia/Baghdad"),
			new TzEntry("IR", "+3540+05126", "Asia/Tehran"),
			new TzEntry("IS", "+6409-02151", "Atlantic/Reykjavik"),
			new TzEntry("IT", "+4154+01229", "Europe/Rome"),
			new TzEntry("JE", "+4912-00207", "Europe/Jersey"),
			new TzEntry("JM", "+1800-07648", "America/Jamaica"),
			new TzEntry("JO", "+3157+03556", "Asia/Amman"),
			new TzEntry("JP", "+353916+1394441", "Asia/Tokyo"),
			new TzEntry("KE", "-0117+03649", "Africa/Nairobi"),
			new TzEntry("KG", "+4254+07436", "Asia/Bishkek"),
			new TzEntry("KH", "+1133+10455", "Asia/Phnom_Penh"),
			new TzEntry("KI", "+0125+17300", "Pacific/Tarawa", "Gilbert Islands"),
			new TzEntry("KI", "-0308-17105", "Pacific/Enderbury", "Phoenix Islands"),
			new TzEntry("KI", "+0152-15720", "Pacific/Kiritimati", "Line Islands"),
			new TzEntry("KM", "-1141+04316", "Indian/Comoro"),
			new TzEntry("KN", "+1718-06243", "America/St_Kitts"),
			new TzEntry("KP", "+3901+12545", "Asia/Pyongyang"),
			new TzEntry("KR", "+3733+12658", "Asia/Seoul"),
			new TzEntry("KW", "+2920+04759", "Asia/Kuwait"),
			new TzEntry("KY", "+1918-08123", "America/Cayman"),
			new TzEntry("KZ", "+4315+07657", "Asia/Almaty", "most locations"),
			new TzEntry("KZ", "+4448+06528", "Asia/Qyzylorda", "Qyzylorda (Kyzylorda, Kzyl-Orda)"),
			new TzEntry("KZ", "+5017+05710", "Asia/Aqtobe", "Aqtobe (Aktobe)"),
			new TzEntry("KZ", "+4431+05016", "Asia/Aqtau",
					"Atyrau (Atirau, Gur'yev), Mangghystau (Mankistau)"),
			new TzEntry("KZ", "+5113+05121", "Asia/Oral", "West Kazakhstan"),
			new TzEntry("LA", "+1758+10236", "Asia/Vientiane"),
			new TzEntry("LB", "+3353+03530", "Asia/Beirut"),
			new TzEntry("LC", "+1401-06100", "America/St_Lucia"),
			new TzEntry("LI", "+4709+00931", "Europe/Vaduz"),
			new TzEntry("LK", "+0656+07951", "Asia/Colombo"),
			new TzEntry("LR", "+0618-01047", "Africa/Monrovia"),
			new TzEntry("LS", "-2928+02730", "Africa/Maseru"),
			new TzEntry("LT", "+5441+02519", "Europe/Vilnius"),
			new TzEntry("LU", "+4936+00609", "Europe/Luxembourg"),
			new TzEntry("LV", "+5657+02406", "Europe/Riga"),
			new TzEntry("LY", "+3254+01311", "Africa/Tripoli"),
			new TzEntry("MA", "+3339-00735", "Africa/Casablanca"),
			new TzEntry("MC", "+4342+00723", "Europe/Monaco"),
			new TzEntry("MD", "+4700+02850", "Europe/Chisinau"),
			new TzEntry("ME", "+4226+01916", "Europe/Podgorica"),
			new TzEntry("MF", "+1804-06305", "America/Marigot"),
			new TzEntry("MG", "-1855+04731", "Indian/Antananarivo"),
			new TzEntry("MH", "+0709+17112", "Pacific/Majuro", "most locations"),
			new TzEntry("MH", "+0905+16720", "Pacific/Kwajalein", "Kwajalein"),
			new TzEntry("MK", "+4159+02126", "Europe/Skopje"),
			new TzEntry("ML", "+1239-00800", "Africa/Bamako"),
			new TzEntry("MM", "+1647+09610", "Asia/Rangoon"),
			new TzEntry("MN", "+4755+10653", "Asia/Ulaanbaatar", "most locations"),
			new TzEntry("MN", "+4801+09139", "Asia/Hovd",
					"Bayan-Olgiy, Govi-Altai, Hovd, Uvs, Zavkhan"),
			new TzEntry("MN", "+4804+11430", "Asia/Choibalsan", "Dornod, Sukhbaatar"),
			new TzEntry("MO", "+2214+11335", "Asia/Macau"),
			new TzEntry("MP", "+1512+14545", "Pacific/Saipan"),
			new TzEntry("MQ", "+1436-06105", "America/Martinique"),
			new TzEntry("MR", "+1806-01557", "Africa/Nouakchott"),
			new TzEntry("MS", "+1643-06213", "America/Montserrat"),
			new TzEntry("MT", "+3554+01431", "Europe/Malta"),
			new TzEntry("MU", "-2010+05730", "Indian/Mauritius"),
			new TzEntry("MV", "+0410+07330", "Indian/Maldives"),
			new TzEntry("MW", "-1547+03500", "Africa/Blantyre"),
			new TzEntry("MX", "+1924-09909", "America/Mexico_City", "Central Time - most locations"),
			new TzEntry("MX", "+2105-08646", "America/Cancun", "Central Time - Quintana Roo"),
			new TzEntry("MX", "+2058-08937", "America/Merida", "Central Time - Campeche, Yucatan"),
			new TzEntry("MX", "+2540-10019", "America/Monterrey",
					"Mexican Central Time - Coahuila, Durango, Nuevo Leon, Tamaulipas away from US border"),
			new TzEntry("MX", "+2550-09730", "America/Matamoros",
					"US Central Time - Coahuila, Durango, Nuevo Leon, Tamaulipas near US border"),
			new TzEntry("MX", "+2313-10625", "America/Mazatlan",
					"Mountain Time - S Baja, Nayarit, Sinaloa"),
			new TzEntry("MX", "+2838-10605", "America/Chihuahua",
					"Mexican Mountain Time - Chihuahua away from US border"),
			new TzEntry("MX", "+2934-10425", "America/Ojinaga",
					"US Mountain Time - Chihuahua near US border"),
			new TzEntry("MX", "+2904-11058", "America/Hermosillo",
					"Mountain Standard Time - Sonora"),
			new TzEntry("MX", "+3232-11701", "America/Tijuana",
					"US Pacific Time - Baja California near US border"),
			new TzEntry("MX", "+3018-11452", "America/Santa_Isabel",
					"Mexican Pacific Time - Baja California away from US border"),
			new TzEntry("MX", "+2048-10515", "America/Bahia_Banderas",
					"Mexican Central Time - Bahia de Banderas"),
			new TzEntry("MY", "+0310+10142", "Asia/Kuala_Lumpur", "peninsular Malaysia"),
			new TzEntry("MY", "+0133+11020", "Asia/Kuching", "Sabah & Sarawak"),
			new TzEntry("MZ", "-2558+03235", "Africa/Maputo"),
			new TzEntry("NA", "-2234+01706", "Africa/Windhoek"),
			new TzEntry("NC", "-2216+16627", "Pacific/Noumea"),
			new TzEntry("NE", "+1331+00207", "Africa/Niamey"),
			new TzEntry("NF", "-2903+16758", "Pacific/Norfolk"),
			new TzEntry("NG", "+0627+00324", "Africa/Lagos"),
			new TzEntry("NI", "+1209-08617", "America/Managua"),
			new TzEntry("NL", "+5222+00454", "Europe/Amsterdam"),
			new TzEntry("NO", "+5955+01045", "Europe/Oslo"),
			new TzEntry("NP", "+2743+08519", "Asia/Kathmandu"),
			new TzEntry("NR", "-0031+16655", "Pacific/Nauru"),
			new TzEntry("NU", "-1901-16955", "Pacific/Niue"),
			new TzEntry("NZ", "-3652+17446", "Pacific/Auckland", "most locations"),
			new TzEntry("NZ", "-4357-17633", "Pacific/Chatham", "Chatham Islands"),
			new TzEntry("OM", "+2336+05835", "Asia/Muscat"),
			new TzEntry("PA", "+0858-07932", "America/Panama"),
			new TzEntry("PE", "-1203-07703", "America/Lima"),
			new TzEntry("PF", "-1732-14934", "Pacific/Tahiti", "Society Islands"),
			new TzEntry("PF", "-0900-13930", "Pacific/Marquesas", "Marquesas Islands"),
			new TzEntry("PF", "-2308-13457", "Pacific/Gambier", "Gambier Islands"),
			new TzEntry("PG", "-0930+14710", "Pacific/Port_Moresby"),
			new TzEntry("PH", "+1435+12100", "Asia/Manila"),
			new TzEntry("PK", "+2452+06703", "Asia/Karachi"),
			new TzEntry("PL", "+5215+02100", "Europe/Warsaw"),
			new TzEntry("PM", "+4703-05620", "America/Miquelon"),
			new TzEntry("PN", "-2504-13005", "Pacific/Pitcairn"),
			new TzEntry("PR", "+182806-0660622", "America/Puerto_Rico"),
			new TzEntry("PS", "+3130+03428", "Asia/Gaza"),
			new TzEntry("PT", "+3843-00908", "Europe/Lisbon", "mainland"),
			new TzEntry("PT", "+3238-01654", "Atlantic/Madeira", "Madeira Islands"),
			new TzEntry("PT", "+3744-02540", "Atlantic/Azores", "Azores"),
			new TzEntry("PW", "+0720+13429", "Pacific/Palau"),
			new TzEntry("PY", "-2516-05740", "America/Asuncion"),
			new TzEntry("QA", "+2517+05132", "Asia/Qatar"),
			new TzEntry("RE", "-2052+05528", "Indian/Reunion"),
			new TzEntry("RO", "+4426+02606", "Europe/Bucharest"),
			new TzEntry("RS", "+4450+02030", "Europe/Belgrade"),
			new TzEntry("RU", "+5443+02030", "Europe/Kaliningrad", "Moscow-01 - Kaliningrad"),
			new TzEntry("RU", "+5545+03735", "Europe/Moscow", "Moscow+00 - west Russia"),
			new TzEntry("RU", "+4844+04425", "Europe/Volgograd", "Moscow+00 - Caspian Sea"),
			new TzEntry("RU", "+5312+05009", "Europe/Samara", "Moscow - Samara, Udmurtia"),
			new TzEntry("RU", "+5651+06036", "Asia/Yekaterinburg", "Moscow+02 - Urals"),
			new TzEntry("RU", "+5500+07324", "Asia/Omsk", "Moscow+03 - west Siberia"),
			new TzEntry("RU", "+5502+08255", "Asia/Novosibirsk", "Moscow+03 - Novosibirsk"),
			new TzEntry("RU", "+5345+08707", "Asia/Novokuznetsk", "Moscow+03 - Novokuznetsk"),
			new TzEntry("RU", "+5601+09250", "Asia/Krasnoyarsk", "Moscow+04 - Yenisei River"),
			new TzEntry("RU", "+5216+10420", "Asia/Irkutsk", "Moscow+05 - Lake Baikal"),
			new TzEntry("RU", "+6200+12940", "Asia/Yakutsk", "Moscow+06 - Lena River"),
			new TzEntry("RU", "+4310+13156", "Asia/Vladivostok", "Moscow+07 - Amur River"),
			new TzEntry("RU", "+4658+14242", "Asia/Sakhalin", "Moscow+07 - Sakhalin Island"),
			new TzEntry("RU", "+5934+15048", "Asia/Magadan", "Moscow+08 - Magadan"),
			new TzEntry("RU", "+5301+15839", "Asia/Kamchatka", "Moscow+08 - Kamchatka"),
			new TzEntry("RU", "+6445+17729", "Asia/Anadyr", "Moscow+08 - Bering Sea"),
			new TzEntry("RW", "-0157+03004", "Africa/Kigali"),
			new TzEntry("SA", "+2438+04643", "Asia/Riyadh"),
			new TzEntry("SB", "-0932+16012", "Pacific/Guadalcanal"),
			new TzEntry("SC", "-0440+05528", "Indian/Mahe"),
			new TzEntry("SD", "+1536+03232", "Africa/Khartoum"),
			new TzEntry("SE", "+5920+01803", "Europe/Stockholm"),
			new TzEntry("SG", "+0117+10351", "Asia/Singapore"),
			new TzEntry("SH", "-1555-00542", "Atlantic/St_Helena"),
			new TzEntry("SI", "+4603+01431", "Europe/Ljubljana"),
			new TzEntry("SJ", "+7800+01600", "Arctic/Longyearbyen"),
			new TzEntry("SK", "+4809+01707", "Europe/Bratislava"),
			new TzEntry("SL", "+0830-01315", "Africa/Freetown"),
			new TzEntry("SM", "+4355+01228", "Europe/San_Marino"),
			new TzEntry("SN", "+1440-01726", "Africa/Dakar"),
			new TzEntry("SO", "+0204+04522", "Africa/Mogadishu"),
			new TzEntry("SR", "+0550-05510", "America/Paramaribo"),
			new TzEntry("ST", "+0020+00644", "Africa/Sao_Tome"),
			new TzEntry("SV", "+1342-08912", "America/El_Salvador"),
			new TzEntry("SY", "+3330+03618", "Asia/Damascus"),
			new TzEntry("SZ", "-2618+03106", "Africa/Mbabane"),
			new TzEntry("TC", "+2128-07108", "America/Grand_Turk"),
			new TzEntry("TD", "+1207+01503", "Africa/Ndjamena"),
			new TzEntry("TF", "-492110+0701303", "Indian/Kerguelen"),
			new TzEntry("TG", "+0608+00113", "Africa/Lome"),
			new TzEntry("TH", "+1345+10031", "Asia/Bangkok"),
			new TzEntry("TJ", "+3835+06848", "Asia/Dushanbe"),
			new TzEntry("TK", "-0922-17114", "Pacific/Fakaofo"),
			new TzEntry("TL", "-0833+12535", "Asia/Dili"),
			new TzEntry("TM", "+3757+05823", "Asia/Ashgabat"),
			new TzEntry("TN", "+3648+01011", "Africa/Tunis"),
			new TzEntry("TO", "-2110-17510", "Pacific/Tongatapu"),
			new TzEntry("TR", "+4101+02858", "Europe/Istanbul"),
			new TzEntry("TT", "+1039-06131", "America/Port_of_Spain"),
			new TzEntry("TV", "-0831+17913", "Pacific/Funafuti"),
			new TzEntry("TW", "+2503+12130", "Asia/Taipei"),
			new TzEntry("TZ", "-0648+03917", "Africa/Dar_es_Salaam"),
			new TzEntry("UA", "+5026+03031", "Europe/Kiev", "most locations"),
			new TzEntry("UA", "+4837+02218", "Europe/Uzhgorod", "Ruthenia"),
			new TzEntry("UA", "+4750+03510", "Europe/Zaporozhye",
					"Zaporozh'ye, E Lugansk / Zaporizhia, E Luhansk"),
			new TzEntry("UA", "+4457+03406", "Europe/Simferopol", "central Crimea"),
			new TzEntry("UG", "+0019+03225", "Africa/Kampala"),
			new TzEntry("UM", "+1645-16931", "Pacific/Johnston", "Johnston Atoll"),
			new TzEntry("UM", "+2813-17722", "Pacific/Midway", "Midway Islands"),
			new TzEntry("UM", "+1917+16637", "Pacific/Wake", "Wake Island"),

			new TzEntry("US", "+404251-0740023", "America/New_York", "Eastern Time"),
			new TzEntry("US", "+421953-0830245", "America/Detroit",
					"Eastern Time - Michigan - most locations"),
			new TzEntry("US", "+381515-0854534", "America/Kentucky/Louisville",
					"Eastern Time - Kentucky - Louisville area"),
			new TzEntry("US", "+364947-0845057", "America/Kentucky/Monticello",
					"Eastern Time - Kentucky - Wayne County"),
			new TzEntry("US", "+394606-0860929", "America/Indiana/Indianapolis",
					"Eastern Time - Indiana - most locations"),
			new TzEntry("US", "+384038-0873143", "America/Indiana/Vincennes",
					"Eastern Time - Indiana - Daviess, Dubois, Knox & Martin Counties"),
			new TzEntry("US", "+410305-0863611", "America/Indiana/Winamac",
					"Eastern Time - Indiana - Pulaski County"),
			new TzEntry("US", "+382232-0862041", "America/Indiana/Marengo",
					"Eastern Time - Indiana - Crawford County"),
			new TzEntry("US", "+382931-0871643", "America/Indiana/Petersburg",
					"Eastern Time - Indiana - Pike County"),
			new TzEntry("US", "+384452-0850402", "America/Indiana/Vevay",
					"Eastern Time - Indiana - Switzerland County"),
			new TzEntry("US", "+415100-0873900", "America/Chicago", "Central Time"),
			new TzEntry("US", "+375711-0864541", "America/Indiana/Tell_City",
					"Central Time - Indiana - Perry County"),
			new TzEntry("US", "+411745-0863730", "America/Indiana/Knox",
					"Central Time - Indiana - Starke County"),
			new TzEntry("US", "+450628-0873651", "America/Menominee",
					"Central Time - Michigan - Dickinson, Gogebic, Iron & Menominee Counties"),
			new TzEntry("US", "+470659-1011757", "America/North_Dakota/Center",
					"Central Time - North Dakota - Oliver County"),
			new TzEntry("US", "+465042-1012439", "America/North_Dakota/New_Salem",
					"Central Time - North Dakota - Morton County (except Mandan area)"),
			new TzEntry("US", "+394421-1045903", "America/Denver", "Mountain Time"),
			new TzEntry("US", "+433649-1161209", "America/Boise",
					"Mountain Time - south Idaho & east Oregon"),
			new TzEntry("US", "+364708-1084111", "America/Shiprock", "Mountain Time - Navajo"),
			new TzEntry("US", "+332654-1120424", "America/Phoenix",
					"Mountain Standard Time - Arizona"),
			new TzEntry("US", "+340308-1181434", "America/Los_Angeles", "Pacific Time"),

			new TzEntry("US", "+611305-1495401", "America/Anchorage", "Alaska Time"),
			new TzEntry("US", "+581807-1342511", "America/Juneau", "Alaska Time - Alaska panhandle"),
			new TzEntry("US", "+593249-1394338", "America/Yakutat",
					"Alaska Time - Alaska panhandle neck"),
			new TzEntry("US", "+643004-1652423", "America/Nome", "Alaska Time - west Alaska"),
			new TzEntry("US", "+515248-1763929", "America/Adak", "Aleutian Islands"),
			new TzEntry("US", "+211825-1575130", "Pacific/Honolulu", "Hawaii"),

			new TzEntry("UY", "-3453-05611", "America/Montevideo"),
			new TzEntry("UZ", "+3940+06648", "Asia/Samarkand", "west Uzbekistan"),
			new TzEntry("UZ", "+4120+06918", "Asia/Tashkent", "east Uzbekistan"),
			new TzEntry("VA", "+415408+0122711", "Europe/Vatican"),
			new TzEntry("VC", "+1309-06114", "America/St_Vincent"),
			new TzEntry("VE", "+1030-06656", "America/Caracas"),
			new TzEntry("VG", "+1827-06437", "America/Tortola"),
			new TzEntry("VI", "+1821-06456", "America/St_Thomas"),
			new TzEntry("VN", "+1045+10640", "Asia/Ho_Chi_Minh"),
			new TzEntry("VU", "-1740+16825", "Pacific/Efate"),
			new TzEntry("WF", "-1318-17610", "Pacific/Wallis"),
			new TzEntry("WS", "-1350-17144", "Pacific/Apia"),
			new TzEntry("YE", "+1245+04512", "Asia/Aden"),
			new TzEntry("YT", "-1247+04514", "Indian/Mayotte"),
			new TzEntry("ZA", "-2615+02800", "Africa/Johannesburg"),
			new TzEntry("ZM", "-1525+02817", "Africa/Lusaka"),
			new TzEntry("ZW", "-1750+03103", "Africa/Harare") };

	/**
	 * @param countryCode
	 *            casing is ignored
	 * @return the set of time zone entries for that country. Set might be
	 *         empty. Never null.
	 */
	public static Set<TzEntry> getEntriesForCountryCode(final String countryCode) {
		final Set<TzEntry> set = new HashSet<TzEntry>();
		for (int i = 0; i < DATABASE.length; i++) {
			if (DATABASE[i].countryCode.equalsIgnoreCase(countryCode)) {
				set.add(DATABASE[i]);
			}
		}
		return set;
	}

	public static TzEntry getEntryForTimeZoneName(final String zoneName) {
		for (int i = 0; i < DATABASE.length; i++) {
			if (DATABASE[i].zoneName.equals(zoneName)) {
				return DATABASE[i];
			}
		}
		return null;
	}
}
