package com.calpano.common.shared.xydrautils.field;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.xydra.base.BaseRuntime;
import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableField;
import org.xydra.base.rmof.XReadableObject;
import org.xydra.base.rmof.XWritableField;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.base.value.XBooleanValue;
import org.xydra.base.value.XIdListValue;
import org.xydra.base.value.XIdSetValue;
import org.xydra.base.value.XIntegerValue;
import org.xydra.base.value.XLongValue;
import org.xydra.base.value.XStringValue;
import org.xydra.base.value.XValue;
import org.xydra.core.model.XField;
import org.xydra.core.model.XObject;

/**
 * A helper to access XValues stored in an {@link XField} of an {@link XObject}.
 *
 * Can handle {@link XStringValue}, {@link XId}, {@link XIdListValue},
 * {@link XIntegerValue}, {@link XLongValue}, and {@link XBooleanValue}.
 *
 * @author xamde
 *
 * @param <T>
 *            a supported Java type or Xydra <b>T</b>ype. Currently supported
 *            Java types are Long, Integer, and Boolean.
 */
public class FieldProperty<T> {

	private final XId fieldId;
	private final Class<T> javaType;

	/**
	 * @param fieldIdString
	 *            e.g. 'name' or 'email'
	 * @param javaType
	 *            must be the same as the generic type of this class. This is
	 *            the only way to have runtime type information.
	 */
	public FieldProperty(final String fieldIdString, final Class<T> javaType) {
		this.fieldId = BaseRuntime.getIDProvider().fromString(fieldIdString);
		assert this.fieldId != null;
		this.javaType = javaType;
	}

	/**
	 * <em>Applies set-like semantics</em>.
	 *
	 * Append 'id' to the list of values. Creates a new list, if required.
	 *
	 * @param actorId
	 *            used to create commands
	 * @param xo
	 *            where to write to
	 * @param id
	 *            appended to list of XID stored in value already
	 * @param avoidDuplicates
	 *            (irrelevant for Sets)
	 */
	@SuppressWarnings("unchecked")
	public void appendXID(final XId actorId, final XObject xo, final XId id, final boolean avoidDuplicates) {
		if (this.javaType.equals(XIdListValue.class)) {
			final XIdListValue currentValue = (XIdListValue) getValue(xo);

			XIdListValue longerXids;
			if (currentValue == null) {
				final XId[] array = new XId[] { id };
				longerXids = BaseRuntime.getValueFactory().createIdListValue(array);
			} else {
				if (avoidDuplicates) {
					// check if already contained
					final List<XId> list = Arrays.asList(currentValue.contents());
					if (list.contains(id)) {
						// nothing to do
						return;
					}
				}
				// else:
				final XId[] longerArray = new XId[currentValue.size() + 1];
				System.arraycopy(currentValue.contents(), 0, longerArray, 0,
						currentValue.contents().length);
				longerArray[currentValue.size()] = id;
				longerXids = BaseRuntime.getValueFactory().createIdListValue(longerArray);
			}
			setValue(actorId, xo, (T) longerXids);
			assert this.getValue(xo) != null;
		} else if (this.javaType.equals(XIdSetValue.class)) {
			final XIdSetValue xidSetValue = (XIdSetValue) getValue(xo);
			XIdSetValue value;
			if (xidSetValue == null || xidSetValue.isEmpty()) {
				value = BaseRuntime.getValueFactory().createIdSetValue(new XId[] { id });
			} else {
				final Set<XId> values = xidSetValue.toSet();
				values.add(id);
				value = BaseRuntime.getValueFactory().createIdSetValue(values);
			}
			setValue(actorId, xo, (T) value);
		} else {
			throw new UnsupportedOperationException("Tpye " + this.javaType
					+ " cannot appendXID. Use XIDListValue or XIDSetValue");
		}

	}

	/**
	 * @param xo
	 *            where to read from
	 * @return the stored value of the field as an instance of type <T>
	 */
	public synchronized T getValue(final XReadableObject xo) {
		return getValueInternal(xo);
	}

	/**
	 * @param xo
	 *            where to read from
	 * @return the stored value of the field as an instance of type <T>
	 */
	private T getValueInternal(final XReadableObject xo) {
		return getValue(xo, this.fieldId, this.javaType);
	}

	/**
	 * @param <T>
	 *            a java type
	 * @param xo
	 *            a non-null object where to read from
	 * @param fieldId
	 *            a non-null XID
	 * @param javaType
	 *            must be of type T; gives method runtime-type information
	 * @return null of an instanceof T
	 */
	@SuppressWarnings("unchecked")
	public static <T> T getValue(final XReadableObject xo, final XId fieldId, final Class<T> javaType) {
		assert xo != null;
		final XReadableField xf = xo.getField(fieldId);

		if (xf == null) {
			// return default as Java would have done
			if (javaType.equals(Boolean.class)) {
				return (T) Boolean.FALSE;
			} else if (javaType.equals(Long.class)) {
				return (T) new Long(0);
			} else if (javaType.equals(Integer.class)) {
				return (T) new Integer(0);
			} else {
				return null;
			}
		}

		final XValue xvalue = xf.getValue();
		if (xvalue == null) {
			throw new NullPointerException("This object (" + xo.getId()
					+ ") needs always to have a value in field '" + fieldId + "'");
		}

		if (xvalue instanceof XStringValue) {
			return (T) ((XStringValue) xvalue).contents();
		} else if (xvalue instanceof XLongValue) {
			return (T) (Long) ((XLongValue) xvalue).contents();
		} else if (xvalue instanceof XIntegerValue) {
			return (T) (Integer) ((XIntegerValue) xvalue).contents();
		} else if (xvalue instanceof XId) {
			return (T) xvalue;
		} else if (xvalue instanceof XIdListValue) {
			return (T) (XIdListValue) xvalue;
		} else if (xvalue instanceof XIdSetValue) {
			return (T) (XIdSetValue) xvalue;
		} else if (xvalue instanceof XBooleanValue) {
			return (T) (Boolean) ((XBooleanValue) xvalue).contents();
		} else {
			// add more cases
			throw new IllegalArgumentException("Cannot handle type '" + xvalue.getClass().getName()
					+ "' (yet). Value = " + xvalue);
		}
	}

	/**
	 * Insert XID at position 0 (the begin)
	 *
	 * @param actorId
	 *            used to create commands
	 * @param xo
	 *            where to write to
	 * @param id
	 *            inserted into list of XID stored in value already
	 * @param avoidDuplicates
	 *            if true, set semantics are used.
	 */
	@SuppressWarnings("unchecked")
	public void insertXID(final XId actorId, final XWritableObject xo, final XId id, final boolean avoidDuplicates) {
		if (this.javaType.equals(XIdListValue.class)) {
			final XIdListValue currentValue = (XIdListValue) getValue(xo);

			XIdListValue longerXids;
			if (currentValue == null) {
				final XId[] array = new XId[] { id };
				longerXids = BaseRuntime.getValueFactory().createIdListValue(array);
			} else {
				if (avoidDuplicates) {
					// check if already contained
					final List<XId> list = Arrays.asList(currentValue.contents());
					if (list.contains(id)) {
						// nothing to do
						return;
					}
				}
				// else:
				final XId[] longerArray = new XId[currentValue.size() + 1];
				System.arraycopy(currentValue.contents(), 0, longerArray, 1,
						currentValue.contents().length);
				longerArray[0] = id;
				longerXids = BaseRuntime.getValueFactory().createIdListValue(longerArray);
			}
			setValue(actorId, xo, (T) longerXids);
			assert this.getValue(xo) != null;
		} else if (this.javaType.equals(XIdSetValue.class)) {
			final XIdSetValue xidSetValue = (XIdSetValue) getValue(xo);
			XIdSetValue value;
			if (xidSetValue == null || xidSetValue.isEmpty()) {
				value = BaseRuntime.getValueFactory().createIdSetValue(new XId[] { id });
			} else {
				final Set<XId> values = xidSetValue.toSet();
				values.add(id);
				value = BaseRuntime.getValueFactory().createIdSetValue(values);
			}
			setValue(actorId, xo, (T) value);
		} else {
			throw new UnsupportedOperationException("Type " + this.javaType
					+ " cannot insertXID. Use XIDListValue or XIDSetValue");
		}

	}

	/**
	 * Set the value of this field within XObject xo to the given value of java
	 * type <T>
	 *
	 * @param actorId
	 *            used to create commands
	 * @param xo
	 *            where to write to
	 * @param value
	 *            to write or null to remove existing value
	 */
	public void setValue(final XId actorId, final XWritableObject xo, final T value) {
		assert xo != null : "xo was null";
		XWritableField xf = xo.getField(this.fieldId);
		if (xf == null) {
			xf = xo.createField(this.fieldId);
		}

		assert xo.hasField(this.fieldId);

		if (value == null) {
			xo.removeField(this.fieldId);
			return;
		}

		XValue xvalue;

		// Java types
		if (value instanceof String) {
			xvalue = BaseRuntime.getValueFactory().createStringValue((String) value);
		} else if (value instanceof Long) {
			xvalue = BaseRuntime.getValueFactory().createLongValue((Long) value);
		} else if (value instanceof Integer) {
			xvalue = BaseRuntime.getValueFactory().createIntegerValue((Integer) value);
		} else if (value instanceof Boolean) {
			xvalue = BaseRuntime.getValueFactory().createBooleanValue((Boolean) value);
		}

		// Xydra types
		else if (value instanceof XId) {
			xvalue = (XId) value;
		} else if (value instanceof XIdListValue) {
			xvalue = (XIdListValue) value;
		} else if (value instanceof XIdSetValue) {
			xvalue = (XIdSetValue) value;
		} else if (value instanceof XId[]) {
			if (this.javaType.equals(XIdSetValue.class)) {
				xvalue = BaseRuntime.getValueFactory().createIdSetValue((XId[]) value);
			} else if (this.javaType.equals(XIdListValue.class)) {
				xvalue = BaseRuntime.getValueFactory().createIdListValue((XId[]) value);
			} else {
				throw new IllegalArgumentException("Cannot handle type '"
						+ value.getClass().getName() + "' for XID[] (yet). Value = " + value);
			}
		} else if (value instanceof Double) {
			xvalue = BaseRuntime.getValueFactory().createDoubleValue((Double) value);
		} else {
			// add more cases
			throw new IllegalArgumentException("Cannot handle type '" + value.getClass().getName()
					+ "' (yet). Value = " + value);
		}
		xf.setValue(xvalue);
	}

	public XId getFieldId() {
		return this.fieldId;
	}

}
