package com.calpano.common.shared.data;

import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.data.DataEvent.DataEventHandler;
import com.google.web.bindery.event.shared.Event;

/**
 * An abstract class to implement {@link DataEvent}s of different
 * {@link com.google.web.bindery.event.shared.Event.Type}s. A {@link DataEvent}
 * is a {@link Event} that happened on a specified data item with a specified
 * {@link ActionKind}, e.g., it indicates that a data item has either
 * successfully been created, loaded or synced, (...) or those actions failed.
 * See {@link ActionKind} to learn what can happen to data.
 *
 * A data event also holds data.
 *
 * @author alpha
 *
 * @param <T> the type of data this DataEvent holds
 */
public abstract class DataEvent<T> extends Event<DataEventHandler<T>> {

	/**
	 * Described the different kinds of {@link DataEvent}s.
	 *
	 * @author alpha
	 */
	public enum ActionKind {

		/** A data entity is available in a DataBridge context */
		// TODO ! Thomas use this instead of more complex Fetch/Load conditions
		// at higher
		// levels
		AvailableSuccess(Result.Success),
		/* Some attempt to make data available failed (e.g. fetch, */
		AvailableFailure(Result.Failure),

		/** A data entity was created */
		CreatedSuccess(Result.Success),
		/** Creation of a data entity failed */
		CreatedFailure(Result.Failure),

		/**
		 * A data entity was fetched from a server (entirely, as opposed to a
		 * mere incremental sync)
		 */
		FetchSuccess(Result.Success),
		/** A complete fetch failed */
		FetchFailure(Result.Failure),

		/* Local init, no data */
		Init(Result.Success),

		/** A data entity was loaded locally */
		LoadSuccess(Result.Success),
		/** A data entity failed to load locally */
		LoadFailure(Result.Failure),

		/** A data entity was stored locally */
		StoreSuccess(Result.Success),
		/** A data entity failed to be stored locally */
		StoreFailure(Result.Failure),

		/** A data entity was synced with a server */
		SyncSuccess(Result.Success),
		/** An attempted sync failed */
		SyncFailure(Result.Failure), ;

		private Result result;

		/**
		 * @param result whether the result of this kind of {@link DataEvent} is
		 *            successful or not
		 */
		ActionKind(final Result result) {
			this.result = result;
		}

		/**
		 * @return true, when the result of this kind of {@link DataEvent} is
		 *         successful
		 */
		public boolean isSuccess() {
			return this.result.equals(Result.Success);
		}
	}

	/**
	 * A {@link DataEventHandler} for {@link DataEvent}s
	 *
	 * @param <T> the type of data item the {@link DataEvent} happened on
	 */
	public interface DataEventHandler<T> {

		/**
		 * Invoked when a {@link DataEvent} happens
		 *
		 * @param event the {@link DataEvent}
		 */
		void onData(DataEvent<T> event);
	}

	/**
	 * Indicates success or failure
	 */
	private enum Result {
		Failure, Success;
	}

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(DataEvent.class);

	private final T data;

	private final ActionKind kind;

	/**
	 * Creates a new {@link DataEvent} that happened on a specified data item
	 * with a specified {@link ActionKind}
	 *
	 * @param kind the {@link ActionKind} of {@link DataEvent}
	 * @param data the data item of that the {@link DataEvent} occurred on
	 */
	public DataEvent(final ActionKind kind, final T data) {
		this.kind = kind;
		this.data = data;
	}

	/**
	 * Creates a new {@link DataEvent} with the specified {@link ActionKind} and
	 * data.
	 *
	 * @param kind the {@link ActionKind} to be represented in this
	 *            {@link DataEvent}
	 * @param data the data in this {@link DataEvent}
	 * @return a new {@link DataEvent} with a specified {@link ActionKind} and
	 *         data item.
	 */
	public abstract DataEvent<T> create(ActionKind kind, T data);

	/**
	 * Creates a copy of this {@link DataEvent} with the specified
	 * {@link ActionKind}. This is used to send a new {@link DataEvent} with a
	 * different kind, e.g., {@link ActionKind#AvailableSuccess}, but the same
	 * {@link DataEvent} type. The type must be same so that the eventbus knows
	 * which listeners to notify. Must be implemented by the concrete
	 * {@link DataEvent}s in order to set the right type.
	 *
	 * @param kind the {@link ActionKind} to be represented in the copy of this
	 *            {@link DataEvent}
	 * @return a copy of this {@link DataEvent} with a specified
	 *         {@link ActionKind} and the same {@link DataEvent} type.
	 */
	public DataEvent<T> createCopyWithKind(final ActionKind kind) {
		return create(kind, getData());
	}

	/**
	 * @return the data item of that the {@link DataEvent} occurred on
	 */
	public T getData() {
		return this.data;
	}

	/**
	 * @return the {@link ActionKind} of {@link DataEvent}
	 */
	public ActionKind getKind() {
		return this.kind;
	}

	@Override
	public String toString() {
		return getClass() + ": " + this.kind.toString();
	}
}
