/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.calpano.common.client.view.forms;

import org.xydra.annotations.LicenseApache;

import com.calpano.common.client.view.resources.CommonResourceBundle;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.event.dom.client.MouseEvent;
import com.google.gwt.safehtml.shared.SafeHtml;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;

/**
 * State-full widget.
 *
 * A tool-tip that can be attached to a widget and follows widget movements. The
 * tooltip can be on on top or below the widget; it can be left, right or
 * centered.
 *
 * Source: http://code.google.com/p/gwt-eye-candy/wiki/TipOfTheDay
 *
 * Apache License 2.0
 *
 * Adapted by Max Völkel
 */
@LicenseApache(project = "gwt-eye-candy")
public class Tooltip {

	public static enum TooltipPosition {
		BELOW_LEFT, BELOW_CENTER, BELOW_RIGHT, ABOVE_LEFT, ABOVE_CENTER, ABOVE_RIGHT, LEFT_TOP, LEFT_MIDDLE, LEFT_BOTTOM, RIGHT_TOP, RIGHT_MIDDLE, RIGHT_BOTTOM
	}

	private final PopupPanel popupPanel = new PopupPanel();

	private final SimplePanel panel = new SimplePanel();

	private final SimplePanel wrapper = new SimplePanel();

	private Widget widget;

	/* internal management to make cancelling delayed showing possible */
	private boolean showing;

	public void hideTooltip() {
		this.showing = false;
		Tooltip.this.popupPanel.hide();
	}

	@SuppressWarnings("unused")
	private static boolean isMouseOutOfWidget(final MouseEvent<?> event, final Widget widget) {
		final int x = event.getClientX();
		final int y = event.getClientY();
		return x <= widget.getAbsoluteLeft()
				|| x >= widget.getAbsoluteLeft() + widget.getOffsetWidth()
				|| y <= widget.getAbsoluteTop()
				|| y >= widget.getAbsoluteTop() + widget.getOffsetHeight();
	}

	private TooltipPosition position = TooltipPosition.BELOW_LEFT;

	public Tooltip() {
		CommonResourceBundle.INSTANCE.css().ensureInjected();
		this.panel.setStyleName(CommonResourceBundle.INSTANCE.css().tooltip());
		this.wrapper.getElement().getStyle().setPosition(Position.RELATIVE);
		this.wrapper.add(this.panel);
		this.popupPanel.add(this.wrapper);
	}

	public Tooltip(final IsWidget content) {
		this();
		this.panel.setWidget(content);
	}

	public Tooltip setContent(final IsWidget content) {
		this.panel.setWidget(content);
		return this;
	}

	public Tooltip setText(final String text) {
		this.panel.setWidget(new Label(text));
		return this;
	}

	public Tooltip setHtml(final SafeHtml html) {
		this.panel.setWidget(new HTML(html));
		return this;
	}

	public Tooltip setPosition(final TooltipPosition position) {
		this.position = position;
		return this;
	}

	public Tooltip attachTo(final Widget widget) {
		// register handlers
		this.widget = widget;
		return this;
	}

	public Tooltip setStyleName(final String style) {
		this.panel.setStyleName(style);
		return this;
	}

	public Tooltip addStyleName(final String style) {
		this.panel.addStyleName(style);
		return this;
	}

	static class Box {
		public int x, y, width, height;

		public Box(final int x, final int y, final int width, final int height) {
			this.x = x;
			this.y = y;
			this.width = width;
			this.height = height;
		}

		public boolean isEqual(final Box other) {
			return this.x == other.x && this.y == other.y && this.width == other.width
					&& this.height == other.height;
		}
	}

	private Box widgetBox = new Box(0, 0, 0, 0);

	static class Pos {
		public Pos(final int x, final int y) {
			super();
			this.x = x;
			this.y = y;
		}

		int x, y;
	}

	protected int popupOffsetWidth;

	protected int popupOffsetHeight;

	final PositionCallback POSITION_CALLBACK = new PositionCallback() {

		@Override
		public void setPosition(final int offsetWidth, final int offsetHeight) {

			Tooltip.this.popupOffsetWidth = offsetWidth;
			Tooltip.this.popupOffsetHeight = offsetHeight;

			Tooltip.this.widgetBox = new Box(Tooltip.this.widget.getAbsoluteLeft(),
					Tooltip.this.widget.getAbsoluteTop(), Tooltip.this.widget.getOffsetWidth(),
					Tooltip.this.widget.getOffsetHeight());

			final Pos pos = Tooltip.this.calcPopupPos(Tooltip.this.widgetBox);
			Tooltip.this.popupPanel.setPopupPosition(pos.x, pos.y);
		}
	};

	public void showTooltipDelayed(final int delay) {
		this.showing = true;
		// try to show in delay ms if we then still want to show it
		Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {
			private boolean ran = false;

			@Override
			public boolean execute() {
				if (!this.ran) {
					this.ran = true;
					return true;
				} else {
					showTooltipTechnically();
					return false;
				}
			}
		}, delay);
	}

	public void showTooltip() {
		this.showing = true;
		showTooltipTechnically();
	}

	private void showTooltipTechnically() {
		if (!this.showing) {
			return;
		}

		this.popupPanel.setPopupPositionAndShow(this.POSITION_CALLBACK);

		// watch widgets pos and size
		Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {

			@Override
			public boolean execute() {
				// check widget
				Tooltip.this.moveTooltipIfNecessary();
				// do until invisible
				return Tooltip.this.panel.isVisible();
			}
		}, 200);
	}

	protected Pos calcPopupPos(final Box widgetBox) {
		int x = 0, y = 0;
		switch (Tooltip.this.position) {
		case ABOVE_CENTER:
			x = widgetBox.x + (widgetBox.width - this.popupOffsetWidth) / 2;
			y = widgetBox.y - this.popupOffsetHeight + 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case ABOVE_LEFT:
			x = widgetBox.x;
			y = widgetBox.y - this.popupOffsetHeight + 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case ABOVE_RIGHT:
			x = widgetBox.x + widgetBox.width - this.popupOffsetWidth;
			y = widgetBox.y - this.popupOffsetHeight + 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case BELOW_CENTER:
			x = widgetBox.x + (widgetBox.width - this.popupOffsetWidth) / 2;
			y = widgetBox.y + widgetBox.height - 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case BELOW_LEFT:
			x = widgetBox.x;
			y = widgetBox.y + widgetBox.height - 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case BELOW_RIGHT:
			x = widgetBox.x + widgetBox.width - this.popupOffsetWidth;
			y = widgetBox.y + widgetBox.height - 1;
			this.panel.setWidth(0.9 * widgetBox.width + "px");
			break;
		case LEFT_TOP:
			x = widgetBox.x - this.popupOffsetWidth + 1;
			y = widgetBox.y;
			break;
		case LEFT_MIDDLE:
			x = widgetBox.x - this.popupOffsetWidth + 1;
			y = widgetBox.y + (widgetBox.height - this.popupOffsetHeight) / 2;
			break;
		case LEFT_BOTTOM:
			x = widgetBox.x - this.popupOffsetWidth + 1;
			y = widgetBox.y + widgetBox.height - this.popupOffsetHeight;
			break;
		case RIGHT_TOP:
			x = widgetBox.x + widgetBox.width - 1;
			y = widgetBox.y;
			break;
		case RIGHT_MIDDLE:
			x = widgetBox.x + widgetBox.width - 1;
			y = widgetBox.y + (widgetBox.height - this.popupOffsetHeight) / 2;
			break;
		case RIGHT_BOTTOM:
			x = widgetBox.x + widgetBox.width - 1;
			y = widgetBox.y + widgetBox.height - this.popupOffsetHeight;
			break;
		}
		return new Pos(x, y);

	}

	protected void moveTooltipIfNecessary() {
		final Box currentBox = new Box(Tooltip.this.widget.getAbsoluteLeft(),
				Tooltip.this.widget.getAbsoluteTop(), Tooltip.this.widget.getOffsetWidth(),
				Tooltip.this.widget.getOffsetHeight());
		if (currentBox.isEqual(this.widgetBox)) {
			// fine
		} else {
			this.widgetBox = currentBox;
			final Pos pos = Tooltip.this.calcPopupPos(this.widgetBox);
			Tooltip.this.popupPanel.setPopupPosition(pos.x, pos.y);
		}
	}

	public void showHTML(final SafeHtml html) {
		if (this.panel.getWidget() instanceof HTML) {
			((HTML) this.panel.getWidget()).setHTML(html);
		} else {
			this.panel.setWidget(new HTML(html));
		}
		showTooltip();
	}

	public void showWidget(final IsWidget widget) {
		this.panel.setWidget(widget);
		showTooltip();
	}

}
