package com.calpano.common.client.util;

import java.util.ArrayList;
import java.util.List;

import org.xydra.annotations.CanBeNull;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.util.SelectableDropDown.SelectableObject;
import com.calpano.common.client.view.ListItemWidget;
import com.calpano.common.client.view.UnorderedListWidget;
import com.calpano.common.client.view.resources.CommonResourceBundle;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.dom.client.Style;
import com.google.gwt.dom.client.Style.Display;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.logical.shared.HasSelectionHandlers;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;

/**
 * To be used, e.g., in an auto-completion drop-down
 *
 * <ul class="dropdowMenu" style="top: 24064px; left: 544px; display: block; ">
 *
 * <li data-value="Alabama" class="active"><a
 * href="#"><strong>A</strong>l<strong
 * >a</strong>b<strong>a</strong>m<strong>a</strong></a></li>
 * <li data-value="Alaska"><a
 * href="#"><strong>A</strong>l<strong>a</strong>sk<strong>a</strong></a></li>
 * <li data-value="Arizona"><a
 * href="#"><strong>A</strong>rizon<strong>a</strong></a></li>
 * <li data-value="Arkansas"><a
 * href="#"><strong>A</strong>rk<strong>a</strong>ns<strong>a</strong>s</a></li>
 *
 * </ul>
 *
 * @author xamde
 * @param <T>
 */
public class SelectableDropDown<T extends SelectableObject> extends Composite implements
		HasSelectionHandlers<T> {

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(SelectableDropDown.class);

	public static interface SelectableObject {
		String getHtmlDisplayString();
	}

	private final UnorderedListWidget ul;

	/**
	 * Must be {@link #positionBelow(Widget)} before {@link #show(List, int)}
	 */
	public SelectableDropDown() {
		CommonResourceBundle.INSTANCE.css().ensureInjected();
		this.ul = new UnorderedListWidget();
		initWidget(this.ul);
	}

	private int currentIndex = 0;

	private final List<T> displayedObjects = new ArrayList<T>();

	@CanBeNull
	public T getCurrentSelection() {
		if (this.displayedObjects.size() == 0) {
			return null;
		}
		if (this.currentIndex == -1) {
			return null;
		}

		return this.displayedObjects.get(this.currentIndex);
	}

	public void hide() {
		getStyleElement().getStyle().setDisplay(Display.NONE);
	}

	public void moveSelectionDown() {
		if (this.displayedObjects.size() == 0) {
			return;
		}

		updateActiveSelection(this.currentIndex,
				(this.currentIndex + 1) % this.displayedObjects.size());
	}

	public void moveSelectionUp() {
		if (this.displayedObjects.size() == 0) {
			return;
		}

		updateActiveSelection(
				this.currentIndex,
				(this.currentIndex - 1 + this.displayedObjects.size())
						% this.displayedObjects.size());
	}

	public void positionBelow(final Widget reference) {
		final int top = reference.getAbsoluteTop() + reference.getOffsetHeight();
		final int left = reference.getAbsoluteLeft();
		addStyleName(CommonResourceBundle.INSTANCE.css().typeahead());
		addStyleName(CommonResourceBundle.INSTANCE.css().dropdownMenu());
		final Style style = getStyleElement().getStyle();
		style.setPosition(Position.ABSOLUTE);
		style.setTop(top, Unit.PX);
		style.setLeft(left, Unit.PX);
		style.setDisplay(Display.BLOCK);
	}

	/**
	 * @param items
	 * @param initialSelected use -1 for none
	 */
	public void show(final List<T> items, final int initialSelected) {
		this.ul.clear();
		this.displayedObjects.clear();

		if (items.size() == 0) {
			hide();
			return;
		}

		int index = 0;
		for (final T item : items) {
			/*
			 * <li data-value="Alabama" class="active"><a
			 * href="#"><strong>A</strong>l<strong
			 * >a</strong>b<strong>a</strong>m<strong>a</strong></a></li>
			 */
			this.displayedObjects.add(item);
			final ListItemWidget li = new ListItemWidget();
			final Anchor a = new Anchor();
			a.addClickHandler(new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					SelectionEvent.fire(SelectableDropDown.this, item);
				}
			});
			final int itemIndex = index;
			a.addMouseOverHandler(new MouseOverHandler() {

				@Override
				public void onMouseOver(final MouseOverEvent event) {
					updateActiveSelection(SelectableDropDown.this.currentIndex, itemIndex);
				}
			});
			a.addMouseOutHandler(new MouseOutHandler() {

				@Override
				public void onMouseOut(final MouseOutEvent event) {
					Scheduler.get().scheduleDeferred(new ScheduledCommand() {

						@Override
						public void execute() {
							updateActiveSelection(itemIndex, -1);
						}
					});
				}
			});
			a.setHTML(item.getHtmlDisplayString());
			li.setWidget(a);
			this.ul.add(li);
			index++;
		}
		updateActiveSelection(-1, initialSelected);
	}

	/**
	 * @param oldIndex -1 for none
	 * @param currentIndex -1 for none
	 */
	private void updateActiveSelection(final int oldIndex, final int currentIndex) {
		// log.info("moving from " + oldIndex + " to " + currentIndex);
		this.currentIndex = currentIndex;

		if (oldIndex >= 0) {
			this.ul.getWidget(oldIndex).removeStyleName(
					CommonResourceBundle.INSTANCE.css().active());
		}
		if (currentIndex >= 0) {
			this.ul.getWidget(currentIndex).addStyleName(
					CommonResourceBundle.INSTANCE.css().active());
		}
	}

	@Override
	public HandlerRegistration addSelectionHandler(final SelectionHandler<T> handler) {
		return this.addHandler(handler, SelectionEvent.getType());
	}

}
