package com.calpano.common.client.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.xydra.base.XAddress;
import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableModel;
import org.xydra.base.rmof.impl.memory.SimpleModel;
import org.xydra.core.XCopyUtils;
import org.xydra.core.model.impl.memory.IMemoryModel;
import org.xydra.core.model.impl.memory.MemoryModel;
import org.xydra.core.serialize.json.JsonParser;
import org.xydra.core.serialize.json.JsonSerializer;
import org.xydra.gwt.store.GwtXydraStoreRestClient;
import org.xydra.store.XydraStore;

import com.calpano.common.client.storage.MemoryModelStorage;
import com.calpano.common.shared.data.DataEvent;
import com.calpano.common.shared.data.DataEvent.ActionKind;
import com.calpano.common.shared.data.DataEvent.DataEventHandler;
import com.calpano.common.shared.data.DataEventCondition;
import com.calpano.common.shared.data.DataItemDefinition;
import com.calpano.common.shared.data.DataSink;
import com.calpano.common.shared.data.Snapshotizer;
import com.calpano.common.shared.data.Snapshotizer.SnapshotizerCallback;
import com.calpano.common.shared.data.Syncer;
import com.calpano.common.shared.data.Syncer.SyncerCallback;
import com.google.web.bindery.event.shared.Event.Type;
import com.google.web.bindery.event.shared.EventBus;

/**
 * Knows how to load and sync multiple models.
 *
 * Allows to state certain expected events and calls a call-back when all
 * expected events occurred.
 *
 *
 * <br>
 * <br>
 * TODO integrate Thomas DataBridge successor that can deal with error
 * conditions
 *
 * TODO ideally, we want to know how fresh data is, i.e. how many minutes and
 * how many revisions old
 *
 * TODO DataManager needs to know about all required models in order to use
 * batch operations over REST
 *
 * TODO needs an actorId, a password, and API endpoint
 *
 * TODO organize sync and store
 *
 */
public class DataManager {

	/**
	 * Callback when all events which are awaited occurred
	 *
	 * @author alpha
	 *
	 */
	public interface AllEventsOccuredCallback {

		/**
		 * Callback when all events occurred
		 */
		void onAllEventsOccured();
	}

	/**
	 * Impl note: This method is overridden for testing in DataManagerPatcher
	 *
	 * @param userId
	 *            TODO use
	 * @param passwordHash
	 *            TODO use
	 * @param restApiLocation
	 *
	 * @return a {@link XydraStore} connected to a remote API via REST
	 */
	public static XydraStore createStore(final XId userId, final String passwordHash, final String restApiLocation) {
		return new GwtXydraStoreRestClient(restApiLocation, new JsonSerializer(), new JsonParser());
	}

	/**
	 * Register a {@link DataEventWatcher} on the given bus for a given set of
	 * events to occur after this watcher is activated. If so, the callback is
	 * called.
	 *
	 * @param eventBus
	 *            @NeverNull
	 * @param callback
	 *            @NeverNull
	 * @param eventConditions
	 *            @NeverNull
	 */
	public static void registerWatcher(final EventBus eventBus, final AllEventsOccuredCallback callback,
			final DataEventCondition... eventConditions) {
		DataEventWatcher.watch(eventBus, callback, eventConditions);
	}

	private final XId actorId;

	/** which data is managed */
	private final Map<XAddress, DataItemDefinition> dataItemDefinitionMap;

	private final DataSink dataSink;

	private final EventBus eventBus;

	private final Map<XAddress, Syncer> modelAddressToSyncerMap;

	private final String passwordHash;

	private final XId repoId;

	private final SnapshotizerCallback snapshotCallback;

	private final Snapshotizer snapshotizer;

	private final XydraStore store;

	private final SyncerCallback syncerCallBack;

	/**
	 * @param actorId
	 * @param repoId
	 * @param passwordHash
	 * @param eventBus
	 * @param dataItemDefinitions
	 *            exact definitions of the handled data items
	 * @param dataSink
	 * @param restApiLocation
	 *            where the rest API can be found
	 */
	public DataManager(final XId actorId, final XId repoId, final String passwordHash, final EventBus eventBus,
			final List<DataItemDefinition> dataItemDefinitions, final DataSink dataSink, final String restApiLocation) {
		this.actorId = actorId;
		this.repoId = repoId;
		this.passwordHash = passwordHash;
		this.eventBus = eventBus;
		this.dataItemDefinitionMap = new HashMap<XAddress, DataItemDefinition>();
		for (final DataItemDefinition dataItemDef : dataItemDefinitions) {
			this.dataItemDefinitionMap.put(dataItemDef.getModelAddress(), dataItemDef);
		}

		this.snapshotCallback = new SnapshotizerCallback() {

			@Override
			public void onFailedSnapshot(final XAddress modelAddress, final Throwable exception) {
				createAndFireDataEvent(modelAddress, ActionKind.FetchFailure, null);
			}

			@Override
			public void onFailedSnapshotizer(final Throwable exception) {
				// TODO the snapshotizer failed as whole. Send events with
				// ActionKind.FetchFailure for all defined data items?
			}

			@Override
			public void onSuccessfulSnapshot(final XReadableModel snapshot) {
				final SimpleModel simpleModel = new SimpleModel(snapshot.getAddress());
				XCopyUtils.copyDataAndRevisions(snapshot, simpleModel);
				final MemoryModel model = new MemoryModel(DataManager.this.actorId,
						DataManager.this.passwordHash, simpleModel);
				/*
				 * Previous reference to this data specified by the event kind
				 * will be orphaned when updating data sink
				 */
				createAndFireDataEvent(snapshot.getAddress(), ActionKind.FetchSuccess, model);
				createSyncerForModelIfNotPresent(model);
			}
		};

		this.syncerCallBack = new SyncerCallback() {

			@Override
			public void onFailedSync(final XAddress modelAddress, final Throwable exception) {
				createAndFireDataEvent(modelAddress, ActionKind.SyncFailure, null);
			}

			@Override
			public void onSuccessfulSync(final IMemoryModel model, final boolean hasChanges) {
				createAndFireDataEvent(model.getAddress(), ActionKind.SyncSuccess, model);
			}
		};

		this.store = createStore(null, null, restApiLocation);
		this.snapshotizer = new Snapshotizer(this.store);
		this.dataSink = dataSink;
		this.modelAddressToSyncerMap = new HashMap<XAddress, Syncer>();

		final List<Type<DataEventHandler<IMemoryModel>>> eventTypeList = new ArrayList<Type<DataEventHandler<IMemoryModel>>>();
		for (final DataItemDefinition itemDef : this.dataItemDefinitionMap.values()) {
			eventTypeList.add(itemDef.getEventType());
		}
		for (final Type<DataEventHandler<IMemoryModel>> eventType : eventTypeList) {
			this.dataSink.registerEventType(eventType);
		}

		/*
		 * TODO should also manage to get data types from local storage first
		 * and then try to get from server for missing types. After that should
		 * invoke synchronizer for all data types?
		 */
	}

	/**
	 * @param modelAddress
	 * @param kind
	 * @param memoryModel
	 *            @CanBeNull
	 */
	private void createAndFireDataEvent(final XAddress modelAddress, final ActionKind kind,
			final IMemoryModel memoryModel) {
		final DataEvent<IMemoryModel> event = this.dataItemDefinitionMap.get(modelAddress)
				.getEventPrototype();
		this.eventBus.fireEvent(event.create(kind, memoryModel));
	}

	private void createSyncerForModelIfNotPresent(final IMemoryModel model) {
		if (!this.modelAddressToSyncerMap.containsKey(model.getAddress())) {
			final Syncer syncer = new Syncer(model, this.store);
			this.modelAddressToSyncerMap.put(model.getAddress(), syncer);
		}
	}

	private void getSnapshots(final List<XAddress> modelAddressList) {
		this.snapshotizer.getModelSnapshots(this.actorId, this.passwordHash, this.snapshotCallback,
				modelAddressList);
	}

	/**
	 * Load all models from localStorage, as requested by local
	 * dataItemDefinitionMap (simply loads multiple keys, one by one).
	 *
	 * Fires events.
	 *
	 * @return locally not found models
	 */
	public List<XAddress> loadDataLocal() {
		final List<XAddress> locallyNotFoundModelAddresses = new ArrayList<XAddress>();

		for (final DataItemDefinition dataItemDef : this.dataItemDefinitionMap.values()) {
			final MemoryModelStorage storage = new MemoryModelStorage(this.actorId, this.repoId,
					this.passwordHash, dataItemDef.getLocalStorageKeyPrefix());
			final IMemoryModel model = storage.get();
			if (model != null) {
				createAndFireDataEvent(model.getAddress(), ActionKind.LoadSuccess, model);
				createSyncerForModelIfNotPresent(model);
			} else {
				final XAddress failedModelAddress = dataItemDef.getModelAddress();
				createAndFireDataEvent(failedModelAddress, ActionKind.LoadFailure, null);
				locallyNotFoundModelAddresses.add(failedModelAddress);
			}
		}

		return locallyNotFoundModelAddresses;
	}

	/**
	 * Loads all data. Tries to get them from local storage first, if not
	 * successful it gets them from the server.
	 *
	 * Loaded data is put into the data sink and events are sent.
	 */
	public void loadDataTryLocalFirstThenRemote() {
		final List<XAddress> locallyNotFoundModelAddresses = loadDataLocal();
		// fetch all snapshots of locally not found models
		getSnapshots(locallyNotFoundModelAddresses);
	}

	public void store(final DataItemDefinition dataItemDef) {
		final IMemoryModel model = this.dataSink.getDataForEventType(dataItemDef.getEventType());
		if (model != null) {
			final MemoryModelStorage storage = new MemoryModelStorage(this.actorId, this.repoId,
					this.passwordHash, dataItemDef.getLocalStorageKeyPrefix());
			storage.store(model);
		}
	}

	/**
	 * Starts the synchronizing process.
	 *
	 * @param dataItemDef
	 */
	public void synchronize(final DataItemDefinition dataItemDef) {
		final Syncer syncer = this.modelAddressToSyncerMap.get(dataItemDef.getModelAddress());
		if (syncer != null) {
			syncer.synchronize(this.syncerCallBack);
		}
	}
}
